% Test Script for noise estimaton using AMP 

%% System Setup
fftSize = 256;
nNullTones = 128;

noiseParams.impPrb = 0.1;
noiseParams.impPwr = 0.25;
noiseParams.bckPwr = 0.0025;

nGampItr = 4;
nTrials = 10;

useComplexNoise = true;

%% Fixed-Point Settings
isVerbose = true; % display under and overflow stats

maxProdWordLen = 100; 
maxSumWordLen = 100;
fpgaMath = fimath('OverflowMode','saturate',...
    'MaxProductWordLength',maxProdWordLen,...
    'MaxSumWordLength',maxSumWordLen);

% template: numerictype(signed,WordLength,FractionalLength)
%Operations
fixPtNumTypes.fftNumericType    = numerictype(true,20,12);
fixPtNumTypes.ifftNumericType   = numerictype(true,16,10);
fixPtNumTypes.expNumericType    = numerictype(false,16,4);
% Variables
fixPtNumTypes.ihat      = numerictype(true,16,15);
fixPtNumTypes.ivar      = numerictype(false,16,18);
fixPtNumTypes.shat      = numerictype(true,16,10);
fixPtNumTypes.svar      = numerictype(false,16,16);
fixPtNumTypes.pvar      = numerictype(false,16,21);
fixPtNumTypes.phat      = numerictype(true,16,16);
fixPtNumTypes.rvar      = numerictype(false,16,18);
fixPtNumTypes.rhat      = numerictype(true,16,14);
fixPtNumTypes.gain      = numerictype(false,16,15);
fixPtNumTypes.LR        = numerictype(false,16,6);
fixPtNumTypes.prbImpgvR = numerictype(false,16,15);

% warning displays (toggle between on and off)
warning off fixed:fi:overflow;
warning off fixed:fi:underflow;

%% Impulse and background noise generation function 
genImpNoise = @(size) genGmNoise(size,struct('nComps',2,...
    'mixVec',[1-noiseParams.impPrb  noiseParams.impPrb ],...
    'vars',[noiseParams.bckPwr noiseParams.bckPwr+noiseParams.impPwr]),...
    useComplexNoise);

%% Select null tones
g3NullTones = [2:23 60:101 156:197 234:255];
%sunNullTones = [1 9 21 33 45 54:76 89 101 113 125];
% nullTones = 1:nNullTones; 
nullTones = g3NullTones;

SINR_gain_float = [];
SINR_gain_fxp = [];

for iTrial = 1:nTrials
    
    %% generate impulse noise in time-domain
    imp = genImpNoise([fftSize 1]);
    frqImp = fft(imp)/sqrt(fftSize);
    
    %% Estimate
    [impEst]        = ampnest(frqImp(nullTones),fftSize,nullTones,noiseParams,nGampItr);
    [impFixPtEst]   = ampnest_fixpt(frqImp(nullTones),fftSize,nullTones,noiseParams,nGampItr,fixPtNumTypes,fpgaMath,isVerbose);
    if ~useComplexNoise % if the noise is real enforce that
        impEst = real(impEst);
    end
    
    %% Ploting
%     mgplot = @(x) plot(abs(x));
%     
%     figure; hold all;
%     mgplot(imp);
%     mgplot(impEst);
%     mgplot(impFixPtEst);
%     legend('Noise', 'Impulse Estimate','Fixed-Point Estimate');
%     
%     pause;
    
    %% Noise Power analysis
    disp(['Actual Noise Power: ' num2str(var(imp))]);
    SINR_gain_float = [SINR_gain_float;-10*log10(var(imp-impEst)/var(imp))];
    SINR_gain_fxp = [SINR_gain_fxp;-10*log10(var(imp-impFixPtEst.double)/var(imp))];
    disp(['Mitigated Noise Power Reduction (floating point): ' num2str(-10*log10(var(imp-impEst)/var(imp)))]);
    disp(['Mitigated Noise Power Reduction (fixed point): ' num2str(-10*log10(var(imp-impFixPtEst.double)/var(imp)))]);

end;

