function [ noiseEst, estHist ] = ampnest( y_n,fftSize,nullTones,noiseParams,nItr )
% Estimates impulse noise from the null tones using the AMP algorithm.
%   Syntax: [NEST] = ampnest(Y_N,NULLTONES,NOISEPARAMS)
%
%   Y_N         - received signal on null tones
%   FFTSIZE     - fft size or number of tones
%   NULLTONES   - list of null tone indeces
%   NOISEPARAMS - noise parameters struct:
%                   impPrb: impulse probability
%                   impPwr: variance of impulsive component
%                   bckPwr: varinace of background noise
%   NITR        - number of AMP iterations
%   NEST        - noise estimate
%   ESTHIST(*)  - estimation history of the implementable parameters

% Intialization
nNulls = length(nullTones);
ihat = zeros(fftSize,1);
ivar = noiseParams.impPrb*noiseParams.impPwr*ones(fftSize,1);
shat = zeros(nNulls,1);

outputHist = (nargout == 2); % we have output history

if outputHist
   estHist = struct(...
       'pvar',zeros(1,nItr),...
       'zhat',zeros(fftSize,nItr),...
       'phat',zeros(nNulls,nItr),...
       'shat',zeros(nNulls,nItr),...
       'svar',zeros(1,nItr),...
       'rvar',zeros(1,nItr),...
       'rhat',zeros(fftSize,nItr),...
       'ifftAugShat',zeros(fftSize,nItr),...
       'expArg',zeros(fftSize,nItr),...
       'gain',zeros(1,nItr),...
       'mgRsq',zeros(fftSize,nItr),...
       'ilrExpTerm',zeros(fftSize,nItr),...
       'expTerm',zeros(fftSize,nItr),...
       'LR',zeros(fftSize,nItr),...
       'prbImpgvR',zeros(fftSize,nItr),...
       'ihat',zeros(fftSize,nItr),...
       'ivar',zeros(fftSize,nItr));
end

% Operations
myfft = @(x,n) fft(x,n)/sqrt(n);
myifft = @(x,n) sqrt(n)*ifft(x,n);

% AMP iterations
for itr=1:nItr
    
    % Output Linear Step
    pvar = sum(ivar)/fftSize;
    zhat = myfft(ihat,fftSize); 
    phat = zhat(nullTones) - pvar.*shat;
    
    shat = (y_n - phat)./(noiseParams.bckPwr + pvar);
    svar = 1./(noiseParams.bckPwr + pvar);
    
    % Input Linear Step
    rvar = fftSize/nNulls/svar;
    augShat = zeros(fftSize,1);
    augShat(nullTones) = shat;
    ifftAugShat = myifft(augShat,fftSize); %of interest for fixed point implementation
    rhat = ihat + rvar.*ifftAugShat;
    
    % Input Nonlinear Step
    gain = noiseParams.impPwr / (noiseParams.impPwr + rvar);
    mgRsq = abs(rhat).^2;
    expArg = mgRsq*gain/rvar;
    expTerm = exp(expArg);
    ilrExpTerm = noiseParams.impPrb/(1-noiseParams.impPrb) * ...
        expTerm; %of interest for fixed point implementation
    LR = rvar/(rvar+noiseParams.impPwr) .* ilrExpTerm;

    prbImpgvR = 1./(1+1./LR+eps); % LR./(LR+1);
    ihat = gain*prbImpgvR.*rhat;
    ivar = prbImpgvR*gain*rvar + gain^2*prbImpgvR.*mgRsq.*(1-prbImpgvR);
    
    if outputHist
        estHist.pvar(itr) = pvar;
        estHist.zhat(:,itr) = zhat;
        estHist.phat(:,itr) = phat;
        estHist.shat(:,itr) = shat;
        estHist.svar(itr) = svar;
        estHist.expArg(:,itr) = expArg;
        estHist.expTerm(:,itr) = expTerm;
        estHist.rvar(itr) = rvar;
        estHist.rhat(:,itr) = rhat;
        estHist.ifftAugShat(:,itr) = ifftAugShat;
        estHist.gain(itr) = gain;
        estHist.mgRsq(:,itr) = mgRsq;
        estHist.ilrExpTerm(:,itr) = ilrExpTerm;
        estHist.LR(:,itr) = LR;
        estHist.prbImpgvR(:,itr) = prbImpgvR;
        estHist.ihat(:,itr) = ihat;
        estHist.ivar(:,itr) = ivar;
    end
    
end

noiseEst = ihat;

end

