function [ noiseEst ] = ampnest_fixpt( y_n,fftSize,nullTones,noiseParams,nItr,fixPtNumericTypes,fpgaMath,isVerbose )
% Estimates impulse noise from the null tones using the AMP algorithm.
%   Syntax: [NEST] = ampnest(Y_N,NULLTONES,NOISEPARAMS)
%
%   Y_N         - received signal on null tones
%   FFTSIZE     - fft size or number of tones
%   NULLTONES   - list of null tone indeces
%   NOISEPARAMS - noise parameters struct:
%                   impPrb: impulse probability
%                   impPwr: variance of impulsive component
%                   bckPwr: varinace of background noise
%   NITR        - number of AMP iterations
%   NEST        - noise estimate

% Choosing 'TrueDouble' will run a floating point version, setting
% 'LoggingMode' to 'off' will disable logging
if isVerbose
    loggingMode = 'on';
else
    loggingMode = 'off';
end
p = fipref('DataTypeOverride', 'ForceOff', ... %'TrueDouble',...
    'LoggingMode',loggingMode);

% Intialization
nNulls  = length(nullTones);
ihat    = zeros(fftSize,1);
ivar    = noiseParams.impPrb*noiseParams.impPwr*ones(fftSize,1);
shat    = zeros(nNulls,1);


% Operations
myifft     = @(x,n) sqrt(n)*ifft(x,n);
myfft      = @(x,n) fft(x,n)/sqrt(n);
myilrexp   = @(x) noiseParams.impPrb/(1-noiseParams.impPrb)*exp(x);

% Variables
ihat        = fi(ihat,fixPtNumericTypes.ihat,fpgaMath);
ivar        = fi(ivar,fixPtNumericTypes.ivar,fpgaMath);
shat        = fi(shat,fixPtNumericTypes.shat,fpgaMath);
pvar        = fi(0,fixPtNumericTypes.pvar,fpgaMath);
phat        = fi(zeros(nNulls,1),fixPtNumericTypes.phat,fpgaMath);
svar        = fi(0,fixPtNumericTypes.svar,fpgaMath);
rvar        = fi(0,fixPtNumericTypes.rvar,fpgaMath);
rhat        = fi(zeros(fftSize,1),fixPtNumericTypes.rhat,fpgaMath);
gain        = fi(0,fixPtNumericTypes.gain,fpgaMath);
LR          = fi(zeros(fftSize,1),fixPtNumericTypes.LR,fpgaMath);
prbImpgvR   = fi(zeros(fftSize,1),fixPtNumericTypes.prbImpgvR,fpgaMath);

% Operation Results
ilrExpResult    = fi(zeros(fftSize,1),fixPtNumericTypes.expNumericType,fpgaMath);
fftResult       = fi(zeros(fftSize,1),fixPtNumericTypes.fftNumericType,fpgaMath);
ifftResult      = fi(zeros(fftSize,1),fixPtNumericTypes.ifftNumericType,fpgaMath);

% AMP iterations
for itr=1:nItr

    pvar(:) = sum(ivar)/fftSize;
    fftResult(:) = myfft(ihat.double,fftSize);
    phat(:) = fftResult(nullTones) - pvar.*shat;
    
    shat(:) = fixPtNumericTypes.shat.divide(y_n-phat,noiseParams.bckPwr+pvar);
    svar(:) = fixPtNumericTypes.svar.divide(1,noiseParams.bckPwr+pvar);
    
    % Input Linear Step
    rvar(:) = fftSize/nNulls/svar;
    augShat = zeros(fftSize,1);
    augShat(nullTones) = shat.double;
    ifftResult(:) = myifft(augShat,fftSize);
    rhat(:) =  ihat + rvar.*ifftResult;
    
    % Input Nonlinear Step
    gain(:) = fixPtNumericTypes.gain.divide(noiseParams.impPwr,noiseParams.impPwr+rvar);
    mgRsq = abs(rhat).^2;
    ilrExpResult(:) = myilrexp(mgRsq.double*gain.double/rvar.double);
    LR(:) = LR.numerictype.divide(rvar,rvar+noiseParams.impPwr)...
        * ilrExpResult;
    prbImpgvR(:) = fixPtNumericTypes.prbImpgvR.divide(LR,LR+1);

    ihat(:) = gain*prbImpgvR.*rhat;
    ivar(:) = prbImpgvR*gain*rvar + ...
        gain^2*prbImpgvR.*mgRsq.*(1-prbImpgvR);

end

if isVerbose
    logreport(ihat,ivar,gain,LR,prbImpgvR,shat,svar,phat,pvar,rhat,rvar,...
        fftResult,ifftResult,ilrExpResult);
end
reset(p);

% return result
noiseEst = ihat;

end

