function [env_data pdf] = RFI_MakeDataClassA (A, Gamma, N, MaxVal)
% [env_data pdf] = RFI_MakeDataClassA (A, Gamma, N, MaxVal)
% Generate sample vector of length N that fit the Class A envelope pdf determined by
% the parameters A, Gamma. The envelope PDF of a Middleton Class A model is
% as follows:
% w(z) =  sigma_{j=1}_{inf} 2 * exp (-A) * A(j-1) / ((j-1)! * rho ^2) * z * exp(-z^2/rho^2)  
% Inputs:  A - Value of the impulsive index parameter of the Middleton Class A model
%          Gamma - Value of the gaussian factor parameter of the Middleton Class A model
%          N - Length of the input data vector
%          MaxVal - Data is generated between the interval [0, MaxVal]
%                   MaxVal should be chosen such that the pdf at that value is almost zero.
% Outputs: pdf - PDF of symmetric alpha stable calculated at N points
%                in the range, [-MaxVal, MaxVal]
%          env_data - N length vector of noise envelope data generated based on
%                     Middleton Class A model
% References:
% [1] S. M. Zabin and H. V. Poor, Efficient estimation of Class A noise parameters
%     via the EM [Expectation-Maximization] algorithms, IEEE Transaction on Information
%     Theory, vol. 37, no. 1, pp. 60-72, Jan. 1991.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Arvind Sujeeth (arvind.sujeeth@mail.utexas.edu)
% 

x = [0:0.001:MaxVal];         % Range of data values
pdf = zeros(size(x));         % Initialize pdf vector
for j = 1: 10
    rho_sq = ((j-1)/A + Gamma )/ (1 + Gamma);
    pdf = pdf + 2 * exp(-1 * A) * (A^(j - 1)) / (factorial(j-1) * (rho_sq))  .* x .* exp (-1 .* (x.^2) / (rho_sq));
end

pdf = pdf./sum(pdf);    % Normalize the PDF
cdf_w = conv(pdf, ones(size(pdf)));     
cdf_w = cdf_w(1:length(pdf));           % Calculate CDF

% Generate N random data based on inverse mapping of the CDF.
env_data = zeros(1,N);
vals = rand(1,N);   % N uniformly distributed random values to be inverse mapped
for i = 1:N
    [cdf_val ind] = find(cdf_w > vals(i));
    if (length(ind) == 0)
        env_data(i) = x(end);
    else
        env_data(i) = x(ind(1));
    end

end

return;