function noise_data = RFI_NoiseSamplesClassA (A, Gamma, N, MaxVal)
%[z w] = RFI_NoiseSamplesClassA (A, Gamma, N, MaxVal, SNR);
%This function generates a vector of synthetic data samples of unit second moment 
%that follow the Class A Middleton Noise model.
% Inputs:   A  - overlap index (in general belongs to [10^-2,1])
%           Gamma - ratio of the intensity of the independent Gaussian
%                   component to the intensity of the impulsive non-Gaussian
%                   component (in general belongs to [10^-6,1])
%           N      - No of samples to be generated. For efficiency of
%                    calculations do not make N greater than 10^3.
%           MaxVal - Data is generated between the interval [-MaxVal, MaxVal]
%                    MaxVal should be chosen such that the pdf at that
%                    value is almost zero.
% Outputs:  noise_data - N length vector of noise data generated based on
%                        the Class A Middleton Noise model with the given
%                        parameters
%
%References:
% [1] A. Spaulding and D. Middleton, Optimum reception in an impulsive
%     interference environment-part I: Coherent detection, IEEE Transactions on 
%     Communications, vol. 25, no. 9, pp. 910923, 1977
% [2] D. Middleton, Non-Gaussian noise models in signal processing
%     for telecommunications: New methods and results for Class A and Class B 
%     noise models, IEEE Transactions on Information Theory, vol. 45, no. 4, 
%     pp. 1129-1149, May 1999.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Marcel Nassar (nassar@ece.utexas.edu)  


x = [linspace(-100, -MaxVal-1, N) linspace(-MaxVal, MaxVal, N) linspace(MaxVal+1, 100, N)]; % Range within which the data is generated
w = zeros(size(x));
%Generate the Pdf using Eq 7 in [1]
for j = 1: 1000 
    rho_sq = ((j-1)/A + Gamma )/ (1 + Gamma); 
    w = w +  exp(-1 * A) * (A^(j - 1)) / (factorial(j-1) * (sqrt(2*pi*rho_sq)))  .*  exp (-1 .* (x.^2) / (2*rho_sq));
end
w = w./sum(w);
cdf_w = conv(w, ones(size(w))); 
cdf_w = cdf_w(1:length(w)); 
vals = rand(1,N);

noise_data=zeros(N); %for efficiency prevents MATLAB from having to reallocating memory every time

for i = 1:N
    [cdf_val ind] = find(cdf_w > vals(i));
    if (length(ind) == 0)
        noise_data(i) = x(end);
    else
        noise_data(i) = x(ind(1));
    end 

end

noise_data=noise_data/sqrt(moment(noise_data,2)); %normalize the samples to have a unit second moment

return; 