function [thePDF, x] = RFI_makeScalarPDFClassA(A, Gamma, NumPDF, MaxVal)
%
% This function generates a vector of synthetic data samples of unit second moment 
% that follow the Class A Middleton Noise model.
% Inputs:   A     - overlap index (in general belongs to [10^-2,1])
%           Gamma - ratio of the intensity of the independent Gaussian
%                   component to the intensity of the impulsive non-Gaussian
%                   component (in general belongs to [10^-6,1])
%           NumPDF - Number of samples of the pdf to output.  This
%                    ends up in a pdf of -100:-Maxval spaced by NumPDF, 
%                    -MaxVal:MaxVal spaced by NumPDF, and MaxVal:100 spaced
%                    by NumPDF for a total of 3*NumPDF points of the
%                    sampled PDF.
%           MaxVal - Data is generated between the interval [-MaxVal, MaxVal]
%                    MaxVal should be chosen such that the pdf at that
%                    value is almost zero.
%
%           Note: Good values for NumPDF are on the order of 1000-2000 with
%                 MaxVal ~ 10-15.           
%
%
% Outputs:  thePDF - discrete pdf generated, use this with
%                    RFI_makeScalarNoiseClassA
%           x      - noise amplitude values corresponding to the output pdf
%
%References:
% [1] A. Spaulding and D. Middleton, Optimum reception in an impulsive
%     interference environment-part I: Coherent detection, IEEE Transactions on 
%     Communications, vol. 25, no. 9, pp. 910923, 1977
% [2] D. Middleton, Non-Gaussian noise models in signal processing
%     for telecommunications: New methods and results for Class A and Class B 
%     noise models, IEEE Transactions on Information Theory, vol. 45, no. 4, 
%     pp. 1129-1149, May 1999.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% 
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Marcel Nassar (nassar@ece.utexas.edu)  
%              Marcus DeYoung (deyoung@ece.utexas.edu)
% 
% Revision history:
% 
% v 1.0: gulati -- initial RFI_NoiseSamplesClassA
% v 1.1beta: mrd -- split into two functions; pdf generation + noise generation
%               for speed improvements.  decoupled the size of the pdf from
%               the number of noise samples.
% v 1.1beta: nassar -- corrected the equation for generating the pdf, it
%               was off by one term in the summation.

x = [linspace(-100, -MaxVal-1, NumPDF) linspace(-MaxVal, MaxVal, NumPDF) ...
     linspace(MaxVal+1, 100, NumPDF)]; % Range within which the data is generated
w = zeros(size(x));


A_over_m = 1; % improve performance and range of usable A
sigma_sq_prime = A*Gamma; % This variable is sigma_sq * (1+gamma), this is done to induce recursion also on 
                        % sigma
z=x*sqrt((1+Gamma)*A); % This is to enable us to use the new modified variable defined above in the exponential of
                   % the recursion
%Evaluate the pdf based on Eq. 7 in [1]
for i=1:1000
    w = w + (A_over_m/sqrt(sigma_sq_prime))*exp(-z.^2/(2*sigma_sq_prime));
    A_over_m = A_over_m * A/i; %improves performance and extends the range of A by recursive implementation.
    sigma_sq_prime = sigma_sq_prime + 1;
end;
w = w*exp(-A)/sqrt(2*pi/((1+Gamma)*A));

thePDF = w./sum(w);
return; 