% RFIdemo_rx_alpha.m is a M-file script that demonstrates the performance of myriad
% filtering in the presence of alpha-stable noise. The demo prooduces a
% plot that compares the communicaiton performance of myriad filtering
% prior to the correlation receiver against weiner filtering and no
% pre-filtering.
%
% NOTE: This program may take a long time to finish depending on the
% settings that you use (no_iterations, etc.)
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers:  Kapil Gulati    (gulati@ece.utexas.edu)
%               Marcel Nassar   (nassar@ece.utexas.edu)
clc; clear;

addpath(genpath('../'))                 %add the directory that contains the required functions

% Settings
SNR=0:5:20;           % SNR range (Note: this is the generalized SNR)
no_bits=10;            % Number of bits for simulation per iteration
M=2;                    % Number of constellation points (currently only 2 is supported)
N=100;                   % Number of sampled points (raised cosine pulse shape, and received signal)
no_iterations = 20;    % Specifies the number of iterations that the code should run
% Parameters of Alpha Stable Noise used in simulations
alpha = 1.5;            % Characteristic Exponent
disper  = 1;            % Dispersion Parameter
delta = 0;              % Localization Parameter

% Initialize arrays to be used
err_rate_alpha_corr         = zeros(1,length(SNR));
err_rate_alpha_myriad_opt   = zeros(1,length(SNR));
err_rate_alpha_myriad_res   = zeros(1,length(SNR));
err_rate_alpha_myriad_low   = zeros(1,length(SNR));
err_rate_alpha_wiener       = zeros(1,length(SNR));


% Pulse Shape Design
y = rcosine(1,5,'default',0.1,10);  % Need to be adjusted to fit N (defined above)
% Need to choose the right parameters according to the value of N
% The value of N should satisfy N=k*2*delay (see help rcosine)
y = y(1:length(y)-1);

% Wiener Filter Design precalculations (autocorrelation)
y_cor = xcorr(y,'unbiased');
y_cor = y_cor(N:length(y_cor));
Ry = toeplitz(y_cor);           % The autocorrelation Matrix of y

% Simulation Part
for j=1:no_iterations

    % generate 100 bits and modulate the signal
    [x_pul x_mod ] = RFI_gen_binary_mod(no_bits,2,y);

    % Symetric Alpha Stable noise generation
    [pdf_alphas alpha_stable] = RFI_MakeDataAlphaStable(alpha, disper, delta, N*no_bits, 1000, 100);

    % Estimate parameters of alpha_stable noise
    delta_est = RFI_EstAlphaS_Localization(alpha_stable);                       % Estimate of the localization parameter
    L = floor(N*no_bits/8);                                                     % No of segments such that data per segement is 8
    alpha_est  = RFI_EstAlphaS_Alpha(alpha_stable, L, delta_est);               % Estimate of the characteristic exponent
    disper_est = RFI_EstAlphaS_Dispersion(alpha_stable, alpha_est, delta_est);  % Estimate of the dispersion parameter


    for i=1:length(SNR)
        Es = (disper*10^(SNR(i)/10));               % Signal Power as per the definition of a generalized SNR.
        r = x_pul.*sqrt(Es) + alpha_stable(:);      % add noise to the signal

        % Correlation Receiver
        [dem_bits err_rate_corr]    = RFI_detect_cor( r, x_mod, N );

        % Wiener Filter Design
        w=inv(Ry+10^(-SNR(i)/10)*eye(N))*y_cor';    %wiener filter coefficients (Solving Wiener-Hopf Equations)
        [dem_bits err_rate_wiener]  = RFI_detect_wiener( r, x_mod, N, w );

        % Myriad Filtering (Implementation Option 0: Fixed resolution)
        [dem_bits err_rate_myriad_res] = RFI_detect_myriad( r, x_mod, N, alpha_est, disper_est, 0 );

        % Myriad Filtering (Implementation Option 1: Polynomial rooting)
        [dem_bits err_rate_myriad_opt] = RFI_detect_myriad( r, x_mod, N, alpha_est, disper_est, 1 );

        % Myriad Filtering (Implementation Option 2: Low Complexity)
        [dem_bits err_rate_myriad_low] = RFI_detect_myriad( r, x_mod, N, alpha_est, disper_est, 2 );


        % Compute the bit error
        err_rate_alpha_corr(i)   = err_rate_alpha_corr(i) + err_rate_corr;
        err_rate_alpha_myriad_res(i) = err_rate_alpha_myriad_res(i) + err_rate_myriad_res;
        err_rate_alpha_myriad_opt(i) = err_rate_alpha_myriad_opt(i) + err_rate_myriad_opt;
        err_rate_alpha_myriad_low(i) = err_rate_alpha_myriad_low(i) + err_rate_myriad_low;
        err_rate_alpha_wiener(i) = err_rate_alpha_wiener(i) + err_rate_wiener;
    end;
    % Display Status
    disp(strcat(num2str(j/no_iterations*100), '% Done'));
end;

err_rate_alpha_corr = err_rate_alpha_corr/no_iterations;
err_rate_alpha_myriad_res = err_rate_alpha_myriad_res/no_iterations;
err_rate_alpha_myriad_opt = err_rate_alpha_myriad_opt/no_iterations;
err_rate_alpha_myriad_low = err_rate_alpha_myriad_low/no_iterations;
err_rate_alpha_wiener = err_rate_alpha_wiener/no_iterations;

plot(SNR,err_rate_alpha_corr, SNR, err_rate_alpha_wiener, SNR, err_rate_alpha_myriad_low, SNR, err_rate_alpha_myriad_res, SNR , err_rate_alpha_myriad_opt );

xlabel('GSNR');ylabel('BER');
title('Communication Performance');
legend('Correlation Receiver','Weiner Filtering', 'Myriad Filtering: Low Complexity Implementation', 'Myriad Filtering: Fixed Resolution Implementation',  'Myriad Filtering: Optimum Implementation' );
