function beta = RFI_myriad_opt( x,k )
%beta = RFI_myriad_opt( x,k );
%This function computes the sample myriad of the window x with linearity
%parameter k. The minimization is performed by expanding the polynomial and
%taking its derivative and setting it to zero. This should find the exact
%solution up to the accuracy of the above operations. 
%
% Inputs:   x                   - the sample window
%           k                   - linearity parameter
%        
% Outputs:  beta                - sample myriad
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Marcel Nassar   (nassar@ece.utexas.edu)
%
% References:
% [1] J.R. Gonzalez and G.R. Arce. "Optimality of the myriad in practical
% impulsive-noise enviroments," IEEE Trans. on Signal Processing, vol 49,
% no.2, pp. 438-441, February 2001. 


%Important notice: The size of x should be small enough not to make the
%objective function overflow

x=x(:);
N=length(x);
x_min = min(x);
x_max = max(x);

% This will provide the necessary coefficients for the expansion of the
% polynomials
% problems can occur depending how u define the x to be a column or row
% vector
lead_coef = ones(N,1);
mid_coef = -2*x;
last_coef = (x.^2 + k^2);

% Constructing the polynomial matrix
polyns = [lead_coef  mid_coef last_coef];

poly_prod = polyns(1,:);
for i=2:N
    poly_prod = conv(poly_prod,polyns(i,:));
end;

% Obtaining the derivative of the constructed polynomial
der_poly_prod = polyder(poly_prod); 
poly_roots = roots(der_poly_prod);
poly_roots = poly_roots(imag(poly_roots)==0); % Choose only the real roots
poly_roots = [poly_roots; x_min; x_max];

% Evaluates the polynomial at the roots of the derivative and the
% extremeties 
poly_values = polyval(poly_prod,poly_roots);
[min_val min_index] = min(poly_values);
beta = poly_roots(min_index);
