function [K_est NIter] = RFI_EMParamK(N, M, z, A, K_init)
% [K_est NIter] = RFI_EMParamK(N, M, z, A, K_init)
% Estimates the parameter K (= impulsive index * Gaussian factor) of a Middleton
% Class A model assuming that the parameter A (impulsive index) is known
% using the Expectation Maximization algorithm developed in [1]
% Inputs:  N - Length of the input data vector
%          M - Number of terms in the infinite summation of the envelope
%              density to consider in the calculations. M = 10 terms were
%              observed to be sufficient to capture most information about the
%              Class A model in most cases.
%          z - Vector of observed envelope values for the noise data
%                     assumed to follow the Middleton Class A model.
%          A      - Value (known) of the parameter A (impulsive index)
%          K_init - Initial value of the parameter K (= impulsive index * Gaussian factor)
%                   (see [1] for details)
% Outputs: K_est  - Estimate of the parameter K ( = impulsive index * Gaussian factor) for the
%                   Middleton Class A model
%          N_iter - Number of iterations taken by the Expectation
%                   Maximization algorithm to converge. (convergence criterion: incremental
%                   fractional error less than 10^-7)
% References:
% [1] S. M. Zabin and H. V. Poor, Efficient estimation of Class A noise parameters
%     via the EM [Expectation-Maximization] algorithms, IEEE Transaction on Information
%     Theory, vol. 37, no. 1, pp. 60-72, Jan. 1991.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Arvind Sujeeth (arvind.sujeeth@mail.utexas.edu)
%

Kp = K_init;                    % Changing variables to relate to equations in [1]
K_prev = 10;                    % Dummy Initialization to run the while loop for the first time
NIter = 0;

while (abs((K_prev - Kp) / K_prev) >  10 ^-7 && NIter < 100)   % convergence criterion, incremental error less than 10^-7.
    % Niter < 100 used to avoid infinite loop which can occur due to inconsistent input envelope data or in-accuracies in calculations

    K_prev = Kp;

    a_ij = zeros (N, M);
    % Calculate a_ij's as defined in equation (8) in [1]
    j = [1:M];
    pi_j = exp(-1*A) * (A.^(j-1)) ./ factorial(j-1);
    for i = 1:N
        for j=1:M
            h_j  = 2 * z(i) * (A + Kp) / (j - 1 + Kp) * exp( -1 * (z(i)^2) * (A + Kp) / (j - 1 + Kp));
            a_ij(i,j) = pi_j(j) * h_j;
        end
        a_ij(i,:) = a_ij(i,:) ./ sum(a_ij(i,:));
    end

    % Equation (32) in [1]
    beta1 = sum((1 + z.^2).*(a_ij(:,1).'));
    beta2 = sum((1 + z.^2) .* ( a_ij(:,2:end) * (1./[1:M-1]).').');
    beta3 = sum(z.^2 .* a_ij(:,1).');
    beta4 = sum((z.^2) .* ( a_ij(:,2:end) * ((1./[1:M-1]).^2).').');
    beta5 = sum((z.^2) .* ( a_ij(:,2:end) * (1./[1:M-1]).').');

    d1 = beta4;
    d2 = 2 * A * beta4 - beta2;
    d3 = beta4 * A^2 - beta2*A + beta3 - beta1 + N;
    d4 = 2 * beta3 * A - beta1 * A;
    d5 = beta3 * A^2;

    r = roots([d1 d2 d3 d4 d5]);
    r_ind = find (r > 9.09 * 10^-7 & r < 1.1 * 10^-2 & imag(r) == 0);
    r_val = r(r_ind);
    % If more than 1 roots are in the valid range of K, then choose the one
    % which yields the maximum value of the objective function
    % If no value of the root is in the range, then quit the iteration with
    % a null vector for K_est
    if (length(r_val) > 1 )
        obj_func = RFICalculateObjFunc (N, M, z, A, K);
        [obj_val obj_ind] = max(obj_func);
        Kp = r_val (obj_ind);
    else
        if (length(r_val) == 1)
            Kp = r_val;
        else
            if (length(r_val) == 0)
                K = []; return;
            end
        end
    end
    NIter = NIter + 1;
end

K_est = Kp;
return
