function [dem_bits err_rate_myriad] = RFI_detect_myriad( corrupted_signal, trans_bits, N, alpha_est, disper_est, option)
%[dem_bits err_rate_myriad] = RFI_detect_myriad(corrupted_signal, trans_bits, N, alpha_est, disper_est)
%This function demodulates the signal by filtering it with a wiener filter
%and then demodulating it with a correlator receiver for Class A interference
%for the binary digital signaling like 2 PAM. It assumes that the constellation is
%symmetric with respect to zero.
%
% Inputs:   corrupted_signal    - the transmitted signal corrupted by symmetric alpha stable noise
%           trans_bits          - the vector of transmitted bits
%           N                   - number of samples per symbol
%           alpha_est           - estimate of the characteristic exponent
%           disper_est          - estimate of the dispersion parameter
%           option              - Selects which myriad filtering implementation to choose:
%                                   1: polynomial rooting method
%                                       - High accuracy
%                                       - Execution time high
%                                   2: Selection Myriad implementation
%                                       - Low complexity and execution time
%                                       - Not optimum but yields good results.
%
% Outputs:  dem_bits            - vector of demodulated bits
%           err_rate_wiener     - the bit error rate using Myriad filtering
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Marcel Nassar  (nassar@ece.utexas.edu)
% ver 1.2 beta: --nassar:   removed option 0 of myriad resolution implementation 
% Set the size of the sliding window. Window size of 8 was observed to give
% good performance
WINDOW_SIZE = 8;
% Allocate space for filtered signal
signal_filtered = zeros(size(corrupted_signal));
% Calculate the factor 'k' for myriad filtering
k = sqrt(alpha_est/(2-alpha_est)) * disper_est^(1/alpha_est);
% Replicate the first value window_size - 1 times for correct sliding
% window operation
signal_appended = [corrupted_signal(1).*ones( WINDOW_SIZE-1, 1); corrupted_signal];

% Sliding window
switch(option)
    case {1} % Polynomial rooting method
        for i = 1:length(signal_appended)-WINDOW_SIZE + 1
            signal_filtered(i) = RFI_myriad_opt((signal_appended(i:i+WINDOW_SIZE-1))',k);
        end

    case {2} % Low complexity method
        for i = 1:length(signal_appended)-WINDOW_SIZE + 1
            signal_filtered(i) = RFI_myriad_apx((signal_appended(i:i+WINDOW_SIZE-1))',k);
        end
end

% Correlation receiver 
[dem_bits err_rate_myriad] = RFI_detect_cor( signal_filtered, trans_bits, N );

return;