function noise_data = RFI_MakeDataBiVarClassA(A, Gamma1, Gamma2, kappa, N)
% noise_data = RFI_MakeDataBivarClassA(A, K1, K2, kappa, N)
% This function generates a N x 2 marix of synthetic noise samples that
% follow the bivariate Middleton Class A model for a two antenna receiver. 
% Inputs:   A  - overlap index (in general belongs to [10^-2,1])
%           Gamma1, Gamma2 - ratio of the intensity of the independent
%                    Gaussian component to the intensity of the impulsive
%                    non-Gaussian component (in general belongs to
%                    [10^-6,1]) at antennas 1 and 2 respectively
%           kappa  - Correlation coefficient between the noise samples at
%                    the two antennas
%           N      - No of samples to be generated. 
% Outputs:  noise_data - N  x 2 matrix of noise data generated based on
%                        the bivariate Middleton Class A Noise model with
%                        the given parameters
%
% References:
% [1]   K.F. McDonald and R.S. Blum. A physically-based impulsive noise
%       model for array observations, Proc. IEEE Asilomar Conference on
%       Signals, Systems& Computers, vol 1, 2-5 Nov. 1997
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Aditya Chopra  (chopra@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 11/20/08  1.2 Beta    gulati    Intial Creation


% Calculate coeffs of covariance matrix K0 and K1
c0 =  sqrt(Gamma1/(1+Gamma1));
c_hat0 = sqrt((Gamma2)/(1+Gamma2));
c1 = sqrt((Gamma1+1/A)/(1+Gamma1));
c_hat1 = sqrt((Gamma2+1/A)/(1+Gamma2));

% Use univaraite normal to generate the bivariate normal with correlation
% coeffecient kappa
theta = 0.5*asin(kappa);
P0 = [c0*cos(theta) c0*sin(theta); c_hat0*sin(theta) c_hat0*cos(theta)];
P1 = [c1*cos(theta) c1*sin(theta); c_hat1*sin(theta) c_hat1*cos(theta)];

X0 = P0*randn(2,N); % Bivariate Gaussian with covariance matrix K0 [1]
X1 = P1*randn(2,N); % Bivariate Gaussian with covariance matrix K1 [2]

% With probability exp(-A) choose a sample from X0 and from X1 with
% probability (1-exp(-A)) 
t = 1*(rand(1,N)<exp(-A));
T =[t;t];
noise_data = (X0.*T + X1.*(1-T))';

return
