function decodedSymbols = RFI_BiVarClassASubOptML4Rx(Y,H,codebook,M,Es,SNR,A,K1,K2,kappa)
% decodedSymbols = RFI_BiVarClassASubOptML4Rx(Y,H,codebook,M,Es,SNR,A,K1,K2,kappa)
% This function implements the four-piece sub-optimal maximum likelihood
% (ML) decoder for a 2x2 MIMO system in the presence of additive bivariate
% Middleton Class A noise developed in [1]
% Inputs:   Y - matrix of the received signal
%           H - 2x2 channel matrix
%           codebook - codebook that contains all the possible 2x1
%                      codewords that could have been transmitted. This
%                      defines the ML search space.
%           M - Modulation order for the QAM constellation used in
%               transmission (M-QAM)
%           Es - Transmit symbol energy
%           SNR - Received signal-to-noise ratio
%           A  - overlap index of the bivariate Middleton Class A noise
%           Gamma1, Gamma2 - ratio of the intensity of the independent
%                    Gaussian component to the intensity of the impulsive
%                    non-Gaussian component (in general belongs to
%                    [10^-6,1]) at antennas 1 and 2 respectively
%           kappa  - Correlation coefficient between the noise samples at
%                    the receiver
% Outputs:  decodedSymbols - Matrix of the ML decoded symbols
%
% References:
% [1] K. Gulati, A. Chopra, R. W. Heath, Jr., B. L. Evans, K. R. Tinsley,
% and X. E. Lin, "MIMO Receiver Design in the Presence of Radio Frequency
% Interference", Proc. IEEE Int. Global Communications Conf., Nov. 30-Dec.
% 4th, 2008, New Orleans, LA USA.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Aditya Chopra  (chopra@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 11/11/08  1.2 Beta    gulati    Initial Implementation

% Parameters of the approximating function to be used.
r = 2.507;a = 0.236;

% Calculate coeffs of covariance matrix K0mat and K1mat
c0 =  sqrt(K1/(1+K1));
c_hat0 = sqrt((K2)/(1+K2));
c1 = sqrt((K1+1/A)/(1+K1));
c_hat1 = sqrt((K2+1/A)/(1+K2));
N_total = (repmat(Y, 1, M^2) - H * codebook)/ sqrt(Es/SNR);
K0mat = [c0^2 c0*c_hat0*kappa;c0*c_hat0*kappa c_hat0^2];
K1mat = [c1^2 c1*c_hat1*kappa;c1*c_hat1*kappa c_hat1^2];
thresh = 2*log(inv(exp(A) -1 )*sqrt(det(K1mat)/det(K0mat)));
const_factor = log(exp(-A)/sqrt(det(K0mat)));

% decoding metric for the real component
N = real(N_total);
P = diag(N.'*(K1mat^-1 - K0mat^-1)*N + thresh);

cost_r = diag(N.'*inv(K0mat)*N)/2 -  const_factor - RFI_ApproxFuncPhiSubOpt(-P/2,r,a);

% decoding metric for the imaginary component
N = imag(N_total);
P = diag(N.'*(K1mat^-1 - K0mat^-1)*N + thresh);

cost_i = diag(N.'*inv(K0mat)*N)/2  - const_factor - RFI_ApproxFuncPhiSubOpt(-P/2,r,a);

% Choose the codeword that maximizes the log-likelihood function using
% two-piece linear approximation of the log-likelihood function
[val ind] = min(cost_r + cost_i);
decodedSymbols = qamdemod(codebook(:,ind),M);

return