function ClassApdf = RFI_MakePDFClassA (X, A, Gamma, M)
% ClassApdf = RFI_MakePDFClassA (X, A, Gamma, M)
% This function evaluates the pdf of class A noise at a vector of values X.
% Inputs:   X     - data points at which the PDF has to be calculated
%           A     - overlap index (in general belongs to [10^-2,1])
%           Gamma - ratio of the intensity of the independent Gaussian
%                  component to the intensity of the impulsive non-Gaussian
%                  component (in general belongs to [10^-6,1])
%           M     - Number of terms in the Class A pdf that should be
%                   considered while generating the noise samples. 
% Outputs:  pdf   - Vector of same size as x containing the PDF of 
%                   Class A interference calculated at the values given in
%                   X                
%
% References:
% [1] A. Spaulding and D. Middleton, Optimum reception in an impulsive
%     interference environment-part I: Coherent detection, IEEE Transactions on
%     Communications, vol. 25, no. 9, pp. 910923, 1977
% [2] D. Middleton, Non-Gaussian noise models in signal processing
%     for telecommunications: New methods and results for Class A and Class B
%     noise models, IEEE Transactions on Information Theory, vol. 45, no. 4,
%     pp. 1129-1149, May 1999.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Marcel Nassar (nassar@ece.utexas.edu)
%              Marcus DeYoung (deyoung@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 09/22/07  1.0         gulati    Original implementation (RFI_ClassApdf)
% 11/22/07  1.1 Beta    nassar    Recursive implementation that improves
%                                 perfomance and increases the range of
%                                 usable parameter A. 
% 11/11/08  1.2 Beta    gulati    New implementation to improve
%                                 computational performance
% 12/4/2008 1.2 Beta    nassar    Generalized the function so that X can be
%                                 a matrix
% 04/02/09  1.2.1       gulati    Fixed the bug where the parameter M was 
%                                 getting reassigned 

ClassApdf = zeros(size(X));
A_over_m = 1;           % to improve performance and range of usable A

for m = 0:M-1
    sigma_sq = (m/A + Gamma) / (1 + Gamma);
    ClassApdf = ClassApdf + A_over_m * normpdf(X,0,sqrt(sigma_sq));
    A_over_m = A_over_m * A / (m+1);        % improves performance and extends the range of A by recursive implementation.
end
ClassApdf = exp(-A) * ClassApdf;

return
