function [A_est NIter] = RFI_EMParamA (N, M, z, A_init, K, epsilon)
% [A_est NIter] = RFI_EMParamA (N, M, z, A_init, K, epsilon)
% Estimates the impulsive index (A) of a Middleton Class A model assuming
% that the parameter K ( = impulsive index * Gaussian factor) is known
% using the Expectation Maximization algorithm developed in [1]
% Inputs:  N - Length of the input data vector
%          M - Number of terms in the infinite summation of the envelope
%              density to consider in the calculations. M = 10 terms were
%              observed to be sufficient to capture most information about the
%              Class A model in most cases.
%          z - Vector of observed envelope values for the noise data
%                     assumed to follow the Middleton Class A model.
%          A_init - Initial value of the parameter A (impulsive index)
%          K  - Value (known) of the parameter K (= impulsive index * Gaussian factor)
%               (see [1] for details)
%          epsilon - To extend the range of the parameter estimates of A.
%                    The estimates of A are assumed to be in the range,
%                    [10^-2 /(1+epsilon), 1+epsilon]
% Outputs: A_est  - Estimate of the impulsive index parameter of the
%                   Middleton Class A model
%          N_iter - Number of iterations taken by the Expectation
%                   Maximization algorithm to converge. (convergence criterion: incremental
%                   fractional error less than 10^-7)
% References:
% [1] S. M. Zabin and H. V. Poor, Efficient estimation of Class A noise parameters
%     via the EM [Expectation-Maximization] algorithms, IEEE Transaction on Information
%     Theory, vol. 37, no. 1, pp. 60-72, Jan. 1991.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Arvind Sujeeth (arvind.sujeeth@mail.utexas.edu)
%

A = A_init;                         % Change of variables to relate it to the reference [1]
A_prev = 10;                        % Dummy Initialization
NIter = 0;

while (abs ((A_prev - A) / A_prev) > (10^-7) && NIter < 100)    % convergence criterion, incremental error less than 10^-7.
    % Niter < 100 used to avoid infinite loop which can occur due to inconsistent input envelope data or in-accuracies in calculations.
    A_prev = A;

    % Calculate a_ij's as defined in equation (8) in [1]
    a_ij = zeros (N, M);
    j = [1:M];
    pi_j = exp(-1*A) * (A.^(j-1)) ./ factorial(j-1);
    for i = 1:N
        for j=1:M
            h_j  = 2 * z(i) * (A + K) / (j - 1 + K) * exp( -1 * (z(i)^2) * (A + K) / (j - 1 + K));
            a_ij(i,j) = pi_j(j) * h_j;
        end
        rtot = sum(a_ij(i,:));
        a_ij(i,:) = a_ij(i,:) ./ rtot;
    end
    beta = sum(a_ij * [0:M-1].');                               % Equation 12, [1]
    phi = sum( ((z.^2).') .* (a_ij *  (1./([K:M-1+K].'))));     % Equation 12, [1]

    a1 = -1 * N - phi;                                          % Equation 15, [1]
    a2 = a1 * K + beta + N;                                     % Equation 15, [1]
    a3 = beta * K;                                              % Equation 15, [1]
    A = (a2 + sqrt(a2^2 - 4*a1*a3))/(-2*a1);                % Equation 16, [1] (corrected)

    if (A  < 10^-2 / ( 1 + epsilon) )                       % Limiting parameter A_est to the range
        A = 10^-2 / (1 + epsilon);                          % [10^-2 / (1+epsilon), (1+epsilon)]
    else
        if (A > 1 + epsilon)
            A = 1 + epsilon;
        end
    end
    NIter = NIter +1;                                       % Increment Number of Iterations
end

A_est = A;                                                  % Estimate of the impulsive index

return;