function BiVarClassA_PDF = RFI_MakePDFBiVarClassA(A, Gamma1, Gamma2, kappa, X)
% BiVarClassA_PDF = RFI_MakePDFBiVarClassA(A, Gamma1, Gamma2, kappa, X)
% This function calculates the probability density function (PDF) of a
% bivariate Middleton Class A noise at a bivariate vector of values X.
% 
% Inputs:   A  - overlap index (in general belongs to [10^-2,1])
%           Gamma1, Gamma2 - ratio of the intensity of the independent
%                    Gaussian component to the intensity of the impulsive
%                    non-Gaussian component (in general belongs to
%                    [10^-6,1]) at antennas 1 and 2 respectively
%           kappa  - Correlation coefficient between the noise samples at
%                    the two antennas
%           X      - N x 2 matrix of data points at which the PDF is
%                    calculated. Each row of X corresponds to the bivariate 
%                    value at which the PDF is calculated.
% Outputs:  BiVarClassA_PDF - A vector of the same length of X containing 
%                             the PDF of the bivariate Middleton Class A
%                             model calculated at the bivariate values 
%                             given in X.
%
% References:
% [1]   K.F. McDonald and R.S. Blum. A physically-based impulsive noise
%       model for array observations, Proc. IEEE Asilomar Conference on
%       Signals, Systems& Computers, vol 1, 2-5 Nov. 1997
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Aditya Chopra  (chopra@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 06/15/09  1.3 Beta    gulati    Initial Creation


% Calculate coeffs of covariance matrix K0 and K1
c0 =  sqrt(Gamma1/(1+Gamma1));
c_hat0 = sqrt((Gamma2)/(1+Gamma2));
c1 = sqrt((Gamma1+1/A)/(1+Gamma1));
c_hat1 = sqrt((Gamma2+1/A)/(1+Gamma2));

K0 = [c0^2 c0*c_hat0*kappa; c0*c_hat0*kappa c_hat0^2];
K1 = [c1^2 c1*c_hat1*kappa; c1*c_hat1*kappa c_hat1^2];

% Evaluate the bivariate Middleton Class A model at the values given in X
BiVarClassA_PDF = exp(-A) * mvnpdf(X, [0 0], K0) + (1 - exp(-A))* mvnpdf(X, [0 0], K1);

return
