function alpha_est = RFI_EstAlphaS_Alpha(noise_data, L, delta_est);
% [alpha_est] = RFI_EstAlphaS_Alpha(noise_data, L, delta_est) 
% Estimates the characteristic exponent (alpha) of the symmetric alpha stable model using the
% method of extreme order statistics as described in [1]
% Inputs:  noise_data - Vector of symmetric alpha stable noise
%          L - Number of segments in which the data vector noise_data has to be divided
%              as defined by equation (2-3) in [1]
%          delta_est - Estimate of the localization parameter of the
%                      symmetric alpha stable model
% 
% Outputs: alpha_est - Estimate of the characteristic exponent of the
%                      symmetric alpha stable model
%
% References:
% [1] G. A. Tsihrintzis and C. L. Nikias, "Fast estimation of the
%     parameters of alpha-stable impulsive interference", IEEE Transactions 
%     on Signal Processing, vol. 44, no 6, pp. 1492-1503, June 1996.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Arvind Sujeeth (arvind.sujeeth@mail.utexas.edu) 

noise_data_cent = noise_data - delta_est;      %  Centralize the data
K = length(noise_data)/L;                      %  Length of each sub-segment of the data

X_max_vec = zeros(1,L); % Initialize a vector to represent the max of each subsegment
X_min_vec = zeros(1,L); % Initialize a vector to represent the min of each subsegment

for i = 1:L
    X_max_l = max(noise_data_cent((i-1)*K + 1 : i*K));
    X_min_l = min(noise_data_cent((i-1)*K + 1 : i*K));
    X_max_vec(i) = log (abs(X_max_l));
    X_min_vec(i) = -1 * log (abs( X_min_l));
end

avg_X_max = sum(X_max_vec)/L; % Average of max vector
avg_X_min = sum(X_min_vec)/L; % Average of min vector
sd_X_max = sqrt ( sum((X_max_vec - avg_X_max).^2) / (L-1));     % Calculate the standard deviation of the max vector, as in eq (2-6) in [1]
sd_X_min = sqrt ( sum((X_min_vec - avg_X_min).^2) / (L-1));     % Calculate the standard deviation of the min vector, as in eq (2-7) in [1]

alpha_est = pi/2/sqrt(6) * (1/sd_X_max + 1/sd_X_min);   % Estimate of alpha as given by equation (2-8) in [1]

return;