function [A_est Gamma1_est Gamma2_est kappa_est] = RFI_MOMBiVarClassAEst(noise_data)
% [A_est Gamma1_est Gamma2_est kappa_est] = RFI_MOMBiVarClassAEst(noise_data)
% This function estimates the parameters of the bivariate Middleton Class A
% model using the method of moments developed in [1]
% Inputs:   noise_data: N x 2 matrix of noise data assumed to follow the
%                       bivariate Middleton Class A model
% Outputs:  A_est:      Estimate of the overlap index parameter of the 
%                       bivariate Middleton Class A model       
%           Gamma1_est: Estimate of the Gaussian factor parameter at
%                       antenna 1 of the bivariate Middleton Class A model
%           Gamma2_est: Estimate of the Gaussian factor parameter at
%                       antenna 1 of the bivariate Middleton Class A model
%           kappa_est:  Estimate of the correlation coefficient between the
%                       noise observation at the two antennas
% 
% References:
% [1] K. Gulati, A. Chopra, R. W. Heath, Jr., B. L. Evans, K. R. Tinsley,
%     and X. E. Lin, "MIMO Receiver Design in the Presence of Radio
%     Frequency Interference", Proc. IEEE Int. Global Communications
%     Conf., Nov. 30-Dec. 4th, 2008, New Orleans, LA USA.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 11/20/08  1.2 Beta    gulati    Initial Creation


% Estimation of moments at the first antenna
m2_1 = moment(noise_data(:,1),2);   % Second order moment
m4_1 = moment(noise_data(:,1),4);   % Fourth order moment
m6_1 = moment(noise_data(:,1),6);   % Sixth order moment

% Perform Estimation at the first antenna (see [1])
% Here: x_1 = K1 / (1 + K1), y_1 = 1/A + K1 / (1 + K1)
a = 15*(3*m2_1^2 - m4_1^2); b = 3*(m6_1 - m2_1*m4_1); c = 5*m4_1^2 - 3*m2_1*m6_1;
x_1 = min(abs(roots([a b c])));
y_1 =( m4_1 - 3*m2_1 * x_1) / (3*(m2_1 - x_1));
% Estimators
Gamma1_est = x_1 / (1 - x_1);
A1_est = (1-x_1)/(y_1-x_1);

% Estimation at the second antenna
m2_2 = moment(noise_data(:,2),2);   % Second order moment
m4_2 = moment(noise_data(:,2),4);   % Fourth order moment
m6_2 = moment(noise_data(:,2),6);   % Sixth order moment

% Perform Estimation at the second antenna (see [1])
% Here: x_2 = K2 / (1 + K2), y_2 = 1/A + K2 / (1 + K2)
a = 15*(3*m2_2^2 - m4_2^2); b = 3*(m6_2 - m2_2*m4_2); c = 5*m4_2^2 - 3*m2_2*m6_2;
x_2 = min(abs(roots([a b c])));
y_2 =( m4_2 - 3*m2_2 * x_2) / (3*(m2_2 - x_2));
% Estimators
Gamma2_est = x_2 / (1 - x_2);
A2_est = (1-x_2)/(y_2-x_2);

% Estimation for kappa
kappa_est = mean(noise_data(:,1).*noise_data(:,2)) * (x_1 - y_1) / (sqrt(x_1*x_2)*(m2_1 - y_1) - sqrt(y_1*y_2)*(m2_1 - x_1));

% Combining estimates of A
A_est = (A1_est + A2_est) / 2;

return;