function [A_est K_est NumIter] = RFI_EMTwoParamEst(N, M, z, A_init, K_init)
% [A_est K_est] = RFI_EMTwoParamEst(N, M, z, A_init, K_init)
% Estimates the parameters of a Middleton Class A model using the Expectation
% Maximization (EM) algorithm developed in [1]. The following parameters are
% estimated : A : Impulsive index
%             K : impulsive index * Gaussian factor
% Inputs:  N - Length of the input data vector
%          M - Number of terms in the infinite summation of the envelope
%              density to consider in the calculations. M = 10 terms were
%              observed to be sufficient to capture most information about the
%              Class A model in most cases.
%          z - Vector of observed envelope values for the noise data
%                     assumed to follow the Middleton Class A model.
%          A_init - Initial value of the parameter A (impulsive index) to
%                   initialize the EM algorithm
%          K_init - Initial value of the parameter K (= impulsive index * Gaussian factor)
%                   (see [1] for details)
% Outputs: A_est  - Estimate of the impulsive index parameter of the
%                   Middleton Class A model
%          K_est  - Estimate of the parameter K (impulsive index * Gaussian factor) of the
%                   Middleton Class A model
%          NumIter- Number of iterations taken by the EM algorithm to
%                   converge
%
% References:
% [1] S. M. Zabin and H. V. Poor, Efficient estimation of Class A noise parameters
%     via the EM [Expectation-Maximization] algorithms, IEEE Transaction on Information
%     Theory, vol. 37, no. 1, pp. 60-72, Jan. 1991.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Arvind Sujeeth (arvind.sujeeth@mail.utexas.edu)
%

% Dummy Initialization
K_prev = 10;
A_prev = 10;
% Change of variables
A = A_init;
Kp = K_init;
% Initialization
NumIter = 0;

while (abs((K_prev - Kp) / K_prev)  + abs((A_prev - A)/A_prev)>  10 ^-7 && NumIter < 100)   % convergence criterion, incremental error less than 10^-7.
    % Niter < 100 used to avoid infinite loop which can occur due to
    % inconsistent input envelope data or in-accuracies in calculations.
    K_prev = Kp; A_prev = A;
    K_possible =[]; A_possible = [];

    % Calculate a_ij's as defined in equation (8) in [1]
    a_ij = zeros (N, M);
    j = [1:M];
    pi_j = exp(-1*A) * (A.^(j-1)) ./ factorial(j-1);
    for i = 1:N
        for j=1:M
            h_j  = 2 * z(i) * (A + Kp) / (j - 1 + Kp) * exp( -1 * (z(i)^2) * (A + Kp) / (j - 1 + Kp));
            a_ij(i,j) = pi_j(j) * h_j;
        end
        a_ij(i,:) = a_ij(i,:) ./ sum(a_ij(i,:));
    end

    beta1 = sum((1 + z.^2).*(a_ij(:,1).'));                             % Appendix, [1]
    beta2 = sum((1 + z.^2) .* ( a_ij(:,2:end) * (1./[1:M-1]).').');     % Appendix, [1]
    beta3 = sum(z.^2 .* a_ij(:,1).');                                   % Appendix, [1]
    beta4 = sum((z.^2) .* ( a_ij(:,2:end) * ((1./[1:M-1]).^2).').');    % Appendix, [1]
    beta5 = sum((z.^2) .* ( a_ij(:,2:end) * (1./[1:M-1]).').');         % Appendix, [1]

    alpha = sum (sum (a_ij.*(repmat([1:M],N,1) - 1)));                  % Equation 10, [1]

    % See Appendix, [1]
    c1 = beta4^2*N^2 - beta2*beta4*N*(beta5 + N) + beta4*N*(beta5 + N)^2;
    c2 = (-beta2*beta4*N^2 - beta2*beta3*beta4*N - beta2*beta4*alpha*N) + (2*beta3*beta4*N - beta1*beta4*N + beta2^2*N + 2*beta4*alpha*N)*(beta5 + N) - (beta2*N)*(beta5 + N)^2;
    c3 = (beta4*N^3 + 2*beta3*beta4*N^2 - beta1*beta4*N^2 - beta1*beta4*N^2 + 2*beta4*alpha*N^2) + (beta3^2*beta4*N + beta2^2*beta3*N - beta1*beta3*beta4*N - beta1*beta4*alpha*N + 2*beta3*beta4*alpha*N + beta4*alpha^2*N) + (-beta2*N^2 - 3*beta2*beta3*N + 2*beta1*beta2*N - beta2*alpha*N)*(beta5 + N) + (N^2 * beta3*N - beta1*N)*(beta5+N)^2;
    c4 = (-2*beta2*beta3*N^2 - 2*beta2*beta3^2*N + 2*beta1*beta2*beta3*N - 2*beta2*beta3*alpha*N) + (2*beta3*N^2 - beta1*N^2 - 3*beta1*beta3*N + 2*beta3^2*N + beta1^2*N - beta1*alpha*N + 2*beta3*alpha*N)*(beta5 + N);
    c5 = (beta3*N^3 - 2*beta1*beta3*N^2 + 2*beta3^2*N^2 + 2*beta3*alpha*N^2) + (beta1^2*beta3*N - 2*beta1*beta3^2*N + beta3^2*N - 2*beta1*beta3*alpha*N + 2*beta3^2*alpha*N + beta3*alpha^2*N);

    r = real(roots([c1 c2 c3 c4 c5]));
    r_ind = find (r > 9.09 * 10^-7 & r < 1.1 * 10^-2 & imag(r) == 0 & real(r) > 0);
    K_r = r(r_ind);                     % Finding roots in range

    j_vec = repmat([1:M], N, 1);
    z_vec = repmat(z.', 1, M);

    K_r = K_r.'; A_corr = [];zeta = [];
    K_r = [K_r 9.09*10^-7 1.1*10^-2];   % Adding boundary K values
    if (length(K_r) > 0)
        for i = 1:length(K_r)
            zeta(i) = sum(sum(z_vec.^2.*a_ij./(j_vec - 1 + K_r(i) )));  % Equation 30, [1]
        end
        A_corr = (N.*K_r + zeta.*K_r -alpha -N - sqrt((N.*K_r + zeta.*K_r -alpha - N ).^2 + 4.*(N+zeta).*alpha.*K_r)) ./ (-2 .* (N + zeta)) ;
        % Corresponding A's for the K's in range

        indA = find(A_corr > 9.09*10^-3 & A_corr < 1.1);
        K_r = K_r(indA); A_corr = A_corr(indA);             % Retain K's and A's in permissible range
    end

    K_possible = [K_possible K_r] ;                     % Add the retained K's to possible candidates for K_est
    A_possible = [A_possible A_corr] ;                  % Add the retained A's to possible candidates for A_est

    % --------------------
    % Estimate A at boundary values of K
    K1 = RFI_EMParamK_AFixed(N,M,z,9.09*10^-3, Kp);     % Estimate A at K = 9.09*10^-3
    K2 = RFI_EMParamK_AFixed(N,M,z,1.1, Kp);            % Estimate A at K = 1.1
    K1 = K1.'; K2 = K2.';
    %--------------------

    % Add to the possible vector of parameters
    if (length(K1) > 0)
        K_possible = [K_possible K1];
        A_possible = [A_possible 9.09*10^-3.*ones(size(K1))];
    end

    if (length(K2) > 0)
        K_possible = [K_possible K2];
        A_possible = [A_possible 1.1.*ones(size(K2))];
    end

    K_possible = [K_possible, 1.1*10^-2, 1.1*10^-2, 9.09*10^-7, 9.09*10^-7];    % Add all boundary values to the possible K parameters
    A_possible = [A_possible, 1.1, 9.09*10^-3, 1.1, 9.09*10^-3];                % Add all boundary values to the possible A parameters

    j_vec = repmat([1:M], N, 1);
    z_vec = repmat(z.', 1, M);
    alpha = sum (sum (a_ij.*(j_vec - 1)));      % Equation 10, [1]

    Obj_func = []; % Calculate the objective function at all possible A/K pairs
    for i = 1:length(K_possible)
        A_i = A_possible(i); K_i = K_possible(i);
        Obj_func(i) = RFI_EMCalculateObjFunc(N,M,z,A_i,K_i);
    end
    [obj_max ind] = max(Obj_func);  % Choose the A/K pair which has the maximum objective function

    Kp = K_possible(ind); A = A_possible(ind);
    NumIter = NumIter + 1;
end

A_est = A; K_est = Kp;
return
