function [noise_data] = RFI_MakeDataAlphaStable(alpha, disper, delta, N, beta)
% [noise_data] = RFI_MakeDataAlphaStable(alpha, disper, delta, N, beta);
% This function generates a vector of synthetic data samples that follow the symmetric alpha
% stable model.
% Inputs:   alpha  - Characteristic Exponent
%           disper - Dispersion Parameter
%           delta  - Localization Parameter
%           N      - No of samples to be generated
%           beta   - Skewness Parameter
% Outputs:  noise_data - N length vector of noise data generated based on
%                        symmetric alpha stable model
%
% Reference:
% [1] Ercan Kuruoglu, "Signal Processing in alpha-stable noise environments
%     - A least lp-NORM approach", PhD Thesis, University of Cambridge, Nov
%     1998.
% [2] Panagiotis Tsakalides, "Array Signal Processing with Alpha Stable
%     Distributions", PhD Thesis, University of Southern California, 1995
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 09/22/07  1.0         gulati    Initial Creation
% 11/14/07  1.1 Beta    gulati    Fixed the fact that S((alpha,disper,0)
%                                 = disper ^ (1/alpha)* S(alpha, 1, 0) in
%                                 the implementation. Now sync with [1].
% 12/05/09  1.4 Beta    gulati    Added skewness parameter beta

%Assigned default value
if nargin < 5
    beta = 0;
end;

clear j;

% Refer to [1],[2] for details on this method of data generation.
if (alpha == 2)         % Gaussian
    noise_data = sqrt(2)*randn(1,N);
elseif (alpha == 1)     % Cauchy
    u = pi*(rand(1,N)-.5*ones(1,N));
    noise_data = 2/pi * ((pi/2 + beta.*u).*tan(u) - beta* log ((pi/2 * (-log (rand(1,N))) .* cos(u))./(pi/2 + beta*u)));
else
    u = pi*(rand(1,N)-.5*ones(1,N));
    u0 = -pi/2*beta*(1-abs(1-alpha))/alpha;
    noise_data = (cos(atan(beta * (tan(pi*alpha/2)))))^(-1/alpha)*(sin (alpha.*(u - u0))) ./ ((cos(u)).^(1/alpha)) .* ((cos((1-alpha).*u + alpha.*u0)) ./ (-log (rand(1,N)))).^((1-alpha)/alpha);
end
%

if (alpha == 1)
    noise_data = (disper)^(1/alpha) .* noise_data + 2/pi * beta  * disper^(1/alpha) * 1/alpha * log (disper) + delta;      % Including dispersion and localization
else
    noise_data = (disper)^(1/alpha) .* noise_data + delta;      % Including dispersion and localization
end

return