function noise_data = RFI_MakeDataClassA (A, Gamma, M, N, NDim)
% noise_data = RFI_MakeDataClassA (A, Gamma, M, N, NDim)
% This function generates a vector of synthetic data samples of unit second moment
% that follow the Class A Middleton Noise model. When NDim > 1, it generates
% data of NDim dimensions based on isotropic extension of the Middleton
% Class A model.
% Inputs:   A  - overlap index (in general belongs to [10^-2,1])
%           Gamma - ratio of the intensity of the independent Gaussian
%                   component to the intensity of the impulsive non-Gaussian
%                   component (in general belongs to [10^-6,1])
%           M      - Number of terms in the Class A pdf that should be
%                    considered while generating the noise samples. 
%           N      - No of samples to be generated. For efficiency of
%                    calculations do not make N greater than 10^3.
% Optional Inputs:
%           NDim   - Dimension of the data being generated (default = 1)
% Outputs:  noise_data - NDim x N size matrix of noise data generated based on
%                        the Class A Middleton Noise model with the given
%                        parameters. 
%
%References:
% [1] A. Spaulding and D. Middleton, Optimum reception in an impulsive
%     interference environment-part I: Coherent detection, IEEE Transactions on
%     Communications, vol. 25, no. 9, pp. 910923, 1977
% [2] D. Middleton, Non-Gaussian noise models in signal processing
%     for telecommunications: New methods and results for Class A and Class B
%     noise models, IEEE Transactions on Information Theory, vol. 45, no. 4,
%     pp. 1129-1149, May 1999.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%              Marcel Nassar (nassar@ece.utexas.edu)
%              Marcus DeYoung (deyoung@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 09/22/07  1.0         gulati    Original RFI_NoiseSamplesClassA
% 11/22/07  1.1 Beta    nassar    Removed the old implementation, and made this
%                                 function just an encapsulation for 
%                                 RFI_makeScalarPDFClassA and RFI_makeScalarNoiseClassA_pdf 
%                                 (written by DeYoung) to improve computational performance.
% 11/11/08  1.2 Beta    gulati    New implementation
% 12/11/10  1.6 Beta    gulati    Extended to generate multidimensional data

if nargin < 5
    NDim = 1;
end

% Generate the weighting factors for the normal density terms 0 through M
m_vec = [0:M-1];
pdf_weights = exp(-A)*A.^m_vec./factorial(m_vec);

% Generate a RV based on the weights calculated. If selectionMat(i) = j,
% then this indicates that the noise_data(j) is derived from the j'th term
% in the Class A density, i.e. normal with variance ((j/A + Gamma / (1 +
% Gamma)). j/N would be approx equal to exp(-A) * A^j/j! (for large N).
selectionMat = randsample([0:M-1],N,true,pdf_weights);

% Generate the noise data based on the selectionMat vector generated
noise_data = zeros(NDim, N);
for m = 0:M-1
    sigma_sq = (m/A + Gamma) / (1 + Gamma);
    inds = find(selectionMat == m);
    noise_data(:, inds) = mvnrnd(zeros(1,NDim), sigma_sq*eye(NDim), length(inds)).';
    %noise_data = noise_data + randn(1,N).*sqrt(sigma_sq).*(selectionMat == m);
end

noise_data = noise_data./repmat(sqrt(mean(noise_data.^2, 2)), 1, N);

return
