function [noise_data GMMstate] = RFI_MakeDataGMM (NDim, muMat, sigmaMat, mixProb, N)
% [noise_data GMMstate] = RFI_MakeDataGMM (NDim, muMat, sigmaMat, mixProb, N)
%This function generates a vector of synthetic data samples that follow the
%Gaussian mixture model (GMM).
% Inputs:   NDim  - Dimensionality of the data samples to be generated
%           muMat - NMixtures x NDim matrix of mean of each Gaussian
%                   mixture component. Each row corresponds to a NDim
%                   vector of mean value of each of the Gaussian
%                   components.
%           sigmaMat - NMixtures x NDim x NDim matrix of covariance
%                      matrices of each Gaussian mixture components. First
%                      index corresponds to the mixture, and second and
%                      third dimensions correspond to the NDim x NDim
%                      covariance matrix for that Gaussian mixture
%                      component.
%          mixProb - NMixtures x 1 vector of mixture probability of each of
%                    of the Gaussian mixture components.
%          N       - Number of data samples to be generated
% Outputs:  noise_data - N x NDim matrix of data samples generated based on
%                        the parameters of the GMM model. Each row
%                        corresponds to a data sample of dimensionality
%                        NDim.
%           GMMstate   - 1 x N vector indicating which Gaussian component
%                        (1:NMixtures) does the corresponding NDim 
%                        dimensional noise sample belong.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 05/28/10  1.5 beta    gulati    Original RFI_MakeDataGMM
% 12/16/10  1.6 beta    gulati    Added the GMM state as an output

noise_data      = zeros(N, NDim);
NMixtures       = length(mixProb);
selectionMat    = randsample([1:NMixtures],N,true,mixProb); % Generate a selection matrix which indicates which component of the GMM is used to generate a particular sample output

% muMat     is  NMixtures x NDim
% sigmaMat  is  NMixtures x NDim x NDim
% noise     is  N x NDim

for i = 1:NMixtures
    % For each of the mixture components, generate data samples at time
    % instances which have that component selected (as per mixture
    % probability, and reflected in selectionMat)
    inds     = find(selectionMat == i);
    MU       = muMat(i,:);                  % MU is then (1 x NDim)
    SIGMA    = squeeze(sigmaMat(i,:,:));    % SIGMA is then (NDim x NDim)
    noise_data(inds,:) = mvnrnd(MU, SIGMA, length(inds));
end

GMMstate = selectionMat;

return
