function [MixProbs MixMeans MixVars] = RFI_EstParamsGMMwithEM (data, NMixtures, NIters)
% [MixProbs MixMeans MixVars] = RFI_EstParamsGMMwithEM (data, NMixtures, NIters)
% This function estimates the parameters of the Gaussian mixture model
% (GMM) using the expectation maximization (EM) algorithm.
% Inputs:       data - NSamples x NDim matrix of data samples used to
%                      estimate the model parameters. Each row corresponds
%                      to a sample value of dimensionality NDim.
%               NMixtures - Number of Gaussian mixtures components to be
%                           estimated.
%               NIters - Number of iterations of the EM algorithm over
%                        which the parameters are estimated.
% Outputs:      MixProbs - NMixtures x 1 vector of estimated mixture
%                          probabilities of the GMM.
%               MixMeans - NMixtures x NDim matrix containing the estimated
%                          mean of each of the Gaussian mixture components.
%                          Each row corresponds to the NDim vector of the
%                          estimated mean of each of the Gaussian
%                          components
%               MixVars  - NMixtures x NDim x NDim containing the estimated
%                          covariance matrices of each of the components of
%                          the GMM. First index corresponds to the mixture,
%                          and second and third dimensions correspond to
%                          the estimated NDim x NDim covariance matrix for
%                          that Gaussian mixture component.
%
% Copyright (c) The University of Texas
% Please see the file Copyright.txt that came with this release for details
% Programmers: Kapil Gulati   (gulati@ece.utexas.edu)
%
% Revisions:
% Date      Version     Owner     Comments
% -------------------------------------------------------------------------
% 05/28/10  1.5 beta    gulati    Original RFI_EstParamsGMMwithEM
% 12/16/10  1.6 beta    gulati    Changed initialization of covariance matrix
%                                 for robustness

% Deduce NSamples and NDim from the size of the input data matrix
NSamples	= size(data,1);
NDim        = size(data,2);

% Initialize mixture probabilities, means, and covariance matrices for the
% EM algorithm
MixProbs    = sort(rand(NMixtures,1),'descend'); MixProbs = MixProbs./sum(MixProbs);
MixMeans    = zeros(NMixtures, NDim);
MixVars     = zeros(NMixtures, NDim, NDim);
for i = 1:NMixtures
    MixVars(i,:,:) = diag(mean(data.^2));
end

% Expectation Maximization algorithm
P_l_X = zeros(NSamples, NMixtures);
for iter = 1:NIters
    
    % For each of the data sample, evaluate the probability of the sample
    % coming from each of the mixture components using current GMM
    % parameters 
    for k  = 1:NMixtures
        if (sum(diag(squeeze(MixVars(k,:,:)))) == 0)    % If variance is zero, then use a very small value for evaluating mvnpdf
            MixVars(k,:,:) = eps * eye(NDim);
        end
        P_l_X(:,k) = MixProbs(k) * mvnpdf(data, MixMeans(k,:), squeeze(MixVars(k,:,:)));
    end
    inds          = find(sum(P_l_X, 2) == 0);               % P_l_X will be NaN at these values
    P_l_X         = P_l_X ./repmat(sum(P_l_X, 2), 1, NMixtures);
    P_l_X(inds,:) = repmat(MixProbs.', length(inds), 1);    % To avoid NaN 
    
    % Update Mixture Probabilities
    MixProbs = mean(P_l_X).';
    
    % For each of the mixture components, update mean and covariance
    % matrices
    for k = 1:NMixtures
        % Update the mean vector
        MixMeans(k,:) = sum(data.*repmat(P_l_X(:,k), 1, NDim))./sum(P_l_X(:,k));
        
        % Update the covariance matrix
        data1 = (data - repmat(MixMeans(k,:), NSamples, 1)).* sqrt(repmat(P_l_X(:,k),1, NDim));
        MixVars(k,:,:) = data1.' * data1 / sum(P_l_X(:,k));
        % Is p = 0, we make variance = 1 so that mvnpdf does not complain.
        % This could be improved !
        if (sum(P_l_X(:,k)) == 0)
            MixVars(k,:,:) = eye(NDim);
        end
    end
    
end


