// PLLSim.cpp: implementation of the PLLSim class.
//
//////////////////////////////////////////////////////////////////////
#include "main.h"
#include "PLLSim.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
void main(int i,char *simfile[])
{
	PLLSim sim;
	if (i>2)
		sim.go(15000,simfile[1],simfile[2]);
	exit(0);
}

static void loadSimFile(char* filename,PLLSim* pll)
{
	fstream f;
	f.open(filename,ios::in|ios::nocreate);
	char buf[255];
	char* out;
	char* value;
	char seps[]="= \t\n";

	//defaults
	double vdd=1.8;
	double cap=100e-12;
	double vco_current_gain=350000;
	while (f.good())
	{
		f.getline(buf,254);
		out = strtok( buf, seps );
		if (out==NULL)
			continue;
		value=strtok( NULL, seps );

		if (!strcmp(out,"vco_file"))
			pll->vco.setup(value);
		if (!strcmp(out,"ref_period"))
			pll->pg.setup(atof(value));
		if (!strcmp(out,"div_ratio"))
			pll->div.setup(atoi(value));
		if (!strcmp(out,"vdd"))
			vdd=atof(value);
		if (!strcmp(out,"cp_file"))
			pll->cp.setup(value);
		if (!strcmp(out,"cap"))
			cap=atof(value);
		if (!strcmp(out,"sim_cycles"))
			pll->iIterations=atoi(value);
		if (!strcmp(out,"vco_current_gain"))
			pll->vco.dCurrentGain=atof(value);
		if (!strcmp(out,"modulation_file"))
			pll->pg.doModulation(value);
		if (!strcmp(out,"start_output"))
			pll->iStartOutput=atoi(value);
		if (!strcmp(out,"delta_period"))
			pll->pg.dDeltaPeriod=atof(value);
		if (!strcmp(out,"delta_phase"))
			pll->pg.dDeltaPhase=atof(value);
		if (!strcmp(out,"modulation_rate"))
			pll->pg.dRate=atof(value);
	}
	pll->filt.setup(vdd,vdd,cap);
	f.close();
}

PLLSim::PLLSim()
{
	cntl_v=filt.dVolts;
	cntl_i=feedback=vco_out=ref=0.0;
	iIterations=1000;	
	iStartOutput=0;
}

PLLSim::~PLLSim()
{

}

void PLLSim::stepRef()
{
	ref=pg.step();
}

void PLLSim::stepFeedback()
{
	vco_out=vco.step(cntl_v,cntl_i);
	feedback=feedback+div.step(vco_out);
}

void PLLSim::go(int num_steps,char *simfile,char* outfile)
{
	double ph=0.0;

	fstream f;
	f.open(outfile,ios::out);
        f.precision(6);

	loadSimFile(simfile,this);

	//Print out Sim parameters to screen
	printf("\nPLL Sim Config\n");
	printf("---------------------------------------\n");
	pg.print();
	printf("---------------------------------------\n");
	div.print();
	printf("---------------------------------------\n");
	vco.print();
	printf("---------------------------------------\n");
	filt.print();
	printf("---------------------------------------\n");
	cp.print();
	printf("---------------------------------------\n");
	
	
	for(int i=0;i<iIterations;i++)
	{
		stepRef();
		stepFeedback();
		cntl_i=0.0;
		if (i%div.div==0)
		{
			ph=pfd.step(this);
			cntl_i=cp.step(ph);
			cntl_v=filt.step(ph,cntl_i);
		}

		if (i>=iStartOutput)
			f << i << " " << vco.dPeriod*2e9 << " " << ph*1e9 << "\n";
	}
}

Divider::Divider()
{
	div=4;
	e_last=0;
}

Divider::Divider(int d)
{
	div=d;
	e_last=0;
}

Divider::~Divider()
{

}

void Divider::setup(int d)
{
	div=d;
}

double Divider::step(double e)
{
	double r=div*(e-e_last);
	e_last=e;
	return r;
}

ChargePump::ChargePump()
{
	//dPhaseStep=10e-12;
	//for (int i=0;i<100;i++)
	//	dI[i]=0.0;
}

ChargePump::~ChargePump()
{
	
}

void ChargePump::setup(char* filename)
{
	oI.load(filename);
}

double ChargePump::step(double p)
{
	return oI.getY(p);
}

Filter::Filter()
{
	dVolts=0.0;
	dCap=200e-12;
}

Filter::~Filter()
{

}

void Filter::setup(double vv,double v,double c)
{
	dVolts=v;
	dCap=c;
	vdd=vv;
}

double Filter::step(double p,double i)
{
	if (i<0)
		i=i*-1;
	dVolts=dVolts+p*i/dCap;
	if (dVolts>vdd)
		dVolts=vdd;
	if (dVolts<0)
		dVolts=0.0;
	return dVolts;
}

VCO::VCO()
{
	iCycle=0;
	dTime=0;
	dPeriod=0;
	dPeriodNom=0;
	dCurrentGain=350000;
}

VCO::~VCO()
{

}

void VCO::setup(char* filename)
{
	oRange.load(filename);
}

double VCO::step(double v,double i)
{
	dPeriodNom=1/oRange.getY(v);
//	if (dTime>50e-6)
//		dPeriodNom=dPeriodNom+.02e-9;
	dPeriod=dPeriodNom-(i/dCurrentGain);
	dTime=dTime+dPeriod;
	iCycle++;
	return dTime;
}

PulseGen::PulseGen()
{
	dPeriod=1e-9;
	dTime=0;
	dDeltaPeriod=0;
	dDeltaPhase=0;
	dRate=200e-6;
	iCycle=0;
	bSSC=0;
}

PulseGen::~PulseGen()
{

}

void PulseGen::print()
{
	printf("Pulse Gen: period=%.6g ns; delta_period=%.6g ns; delta_phase=%.6g ns; mod_rate=%.6g s\n",
		dPeriod*1e9,dDeltaPeriod*1e9,dDeltaPhase*1e9,dRate);
	if (bSSC)
	{
		printf("Modulation Table:\n");
		oSSC.print();
	}
	else
		printf("No Modulation Table defined.\n");
}
void PulseGen::setup(double period)
{
	dPeriod=period;
	dPeriodNom=period;
}

void PulseGen::doModulation(char* modfile)
{
	oSSC.load(modfile);
	bSSC=1;
}

double PulseGen::step()
{
        if (bSSC)
        {
        	double xmax=oSSC.xx[oSSC.size-1];
        	double xnorm=dTime-int(dTime/xmax)*xmax;
        	dPeriod=dPeriodNom*oSSC.getY(xnorm);
	}
	if (dTime>dRate && dDeltaPeriod!=0)
	{
		dPeriod=dPeriod+dDeltaPeriod;
		dDeltaPeriod=0;
	}

	if (dTime>dRate && dDeltaPhase!=0)
	{
		dTime=dTime+dDeltaPhase;
		dDeltaPhase=0;
	}
	iCycle++;
	dTime=dTime+dPeriod;
	return dTime;
}

PFD::PFD()
{
	e1=e2=0;	
}

PFD::~PFD()
{

}

double PFD::step(PLLSim* pll)
{
	double ph=pll->feedback-pll->ref;  //positive means ref is leading
	int s=0;
	if (ph>=0)
	{
		s=1;
		while (pll->ref<=e2)
		{
			pll->stepRef();
		}
		e2=pll->feedback;
		e1=pll->ref;
	}
	else	
	{
		s=-1;
		while (pll->feedback<=e2)
		{
			pll->stepFeedback();
		}
		e1=pll->feedback;
		e2=pll->ref;
	}
	ph=s*(e2-e1);
	return ph;
}

void InterpolateArray::print()
{
	for (int a=0;a<size;a++)
		printf("%10d  %12.6g %12.6g\n",a,xx[a],yy[a]);

}

double InterpolateArray::getY(double x)
{
	int s=1;
	if (x<0.0)
	{
		s=-1;
		x=-x;
	}

	int i=-1;
	if (x<=xx[0])
		i=0;
	if (x>=xx[size-1])
		i=size-2;
	if (i==-1)
	{
		i++;
		while (x>xx[i])
			i++;
		i--;
	}
	double m=(yy[i]-yy[i+1])/(xx[i]-xx[i+1]);
	return s*(m*(x-xx[i])+yy[i]);
}
void InterpolateArray::load(char* filename)
{
	fstream f;
	f.open(filename,ios::in|ios::nocreate);
	char buf[255];
	char* out;
	char seps[]=" \t\n";

	size=0;
	while (f.good())
	{
		f.getline(buf,254);
		out = strtok( buf, seps );
		if (out!=NULL)
			size++;
	}
	f.close();
	xx=new double[size];
	yy=new double[size];

	f.open(filename,ios::in|ios::nocreate);
	
	int i=0;
	while (f.good())
	{
		f.getline(buf,254);
		out = strtok( buf, seps );
		if (out==NULL)
			continue;
		xx[i]=atof(out);
		out=strtok( NULL, seps );
		if (out==NULL)
		{
			yy[i]=0.0;
			continue;
		}
		else
			yy[i]=atof(out);
		i++;
	}
	f.close();
}

